<?php
// Database setup script - Run this once to create required tables
session_start();
if (!isset($_SESSION['admin_logged_in'])) {
    die('Access denied. Please login first.');
}

require_once '../app/config/database.php';
$database = new Database();
$db = $database->getConnection();

$tables = [
    // Posts table
    "CREATE TABLE IF NOT EXISTS posts (
        id INT AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255) NOT NULL,
        content TEXT NOT NULL,
        category_id INT NULL,
        status ENUM('draft', 'published') DEFAULT 'draft',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )",
    
    // Pages table
    "CREATE TABLE IF NOT EXISTS pages (
        id INT AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255) NOT NULL,
        slug VARCHAR(255) UNIQUE NOT NULL,
        content TEXT NOT NULL,
        status ENUM('draft', 'published') DEFAULT 'draft',
        meta_title VARCHAR(255) NULL,
        meta_description TEXT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )",
    
    // Categories table
    "CREATE TABLE IF NOT EXISTS categories (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL,
        description TEXT NULL,
        type ENUM('post', 'course', 'general') DEFAULT 'post',
        status ENUM('active', 'inactive') DEFAULT 'active',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )",
    
    // Media table
    "CREATE TABLE IF NOT EXISTS media (
        id INT AUTO_INCREMENT PRIMARY KEY,
        filename VARCHAR(255) NOT NULL,
        original_name VARCHAR(255) NOT NULL,
        file_path VARCHAR(500) NOT NULL,
        file_size INT NOT NULL,
        mime_type VARCHAR(100) NOT NULL,
        uploaded_by VARCHAR(100) NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )",
    
    // Settings table
    "CREATE TABLE IF NOT EXISTS settings (
        id INT AUTO_INCREMENT PRIMARY KEY,
        setting_key VARCHAR(100) UNIQUE NOT NULL,
        setting_value TEXT NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )",
    
    // Users table (for user management)
    "CREATE TABLE IF NOT EXISTS users (
        id INT AUTO_INCREMENT PRIMARY KEY,
        email VARCHAR(255) UNIQUE NOT NULL,
        password VARCHAR(255) NOT NULL,
        first_name VARCHAR(100),
        last_name VARCHAR(100),
        current_level ENUM('A1', 'A2', 'B1', 'B2', 'C1', 'C2'),
        test_score INT,
        learning_path JSON,
        status ENUM('active', 'inactive', 'suspended') DEFAULT 'active',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )"
];

echo "<div class='container mt-5'>";
echo "<h2>Database Setup</h2>";
echo "<div class='card'>";
echo "<div class='card-body'>";

foreach ($tables as $table) {
    try {
        $db->exec($table);
        echo "<div class='alert alert-success'>✓ Table created successfully</div>";
    } catch (PDOException $e) {
        echo "<div class='alert alert-warning'>⚠ Table already exists or error: " . $e->getMessage() . "</div>";
    }
}

// Insert default settings
$default_settings = [
    'site_title' => 'Simturax English Learning',
    'site_description' => 'Learn English online with personalized courses',
    'admin_email' => 'admin@simturax.com',
    'items_per_page' => '10',
    'maintenance_mode' => '0',
    'registration_enabled' => '1'
];

foreach ($default_settings as $key => $value) {
    try {
        $checkQuery = "SELECT COUNT(*) as count FROM settings WHERE setting_key = ?";
        $checkStmt = $db->prepare($checkQuery);
        $checkStmt->execute([$key]);
        $result = $checkStmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result['count'] == 0) {
            $query = "INSERT INTO settings (setting_key, setting_value) VALUES (?, ?)";
            $stmt = $db->prepare($query);
            $stmt->execute([$key, $value]);
            echo "<div class='alert alert-info'>✓ Default setting added: $key</div>";
        }
    } catch (PDOException $e) {
        echo "<div class='alert alert-warning'>⚠ Error adding setting $key: " . $e->getMessage() . "</div>";
    }
}

echo "<div class='alert alert-success mt-3'><strong>Database setup completed!</strong></div>";
echo "<a href='index.php' class='btn btn-primary'>Go to Admin Panel</a>";
echo "</div></div></div>";
?>