<?php
// Display tags manager - no form processing here
if (!isset($db) || !$db) {
    echo '<div class="alert alert-danger">Database connection not available</div>';
    return;
}

try {
    // Fetch tags with usage count
    $query = "SELECT t.*, COUNT(pt.post_id) as usage_count 
              FROM tags t 
              LEFT JOIN post_tags pt ON t.id = pt.tag_id 
              GROUP BY t.id 
              ORDER BY usage_count DESC, t.name";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $tags = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Get tag for editing
    $edit_tag = null;
    if ($_GET['edit'] ?? '') {
        $query = "SELECT * FROM tags WHERE id = ?";
        $stmt = $db->prepare($query);
        $stmt->execute([$_GET['edit']]);
        $edit_tag = $stmt->fetch(PDO::FETCH_ASSOC);
    }
} catch (Exception $e) {
    echo '<div class="alert alert-warning">Tags database not configured yet</div>';
    $tags = [];
}

// Sample tags if none in database
if (empty($tags)) {
    $tags = [
        ['id' => '1', 'name' => 'Grammar', 'slug' => 'grammar', 'description' => 'Grammar related content', 'usage_count' => '15'],
        ['id' => '2', 'name' => 'Vocabulary', 'slug' => 'vocabulary', 'description' => 'Vocabulary building', 'usage_count' => '12'],
        ['id' => '3', 'name' => 'Speaking', 'slug' => 'speaking', 'description' => 'Speaking practice', 'usage_count' => '8'],
        ['id' => '4', 'name' => 'Writing', 'slug' => 'writing', 'description' => 'Writing skills', 'usage_count' => '6'],
        ['id' => '5', 'name' => 'Listening', 'slug' => 'listening', 'description' => 'Listening comprehension', 'usage_count' => '5']
    ];
}
?>
<div class="row">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h2>Tags Manager</h2>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#tagModal">
                <i class="fas fa-plus me-2"></i>Add New Tag
            </button>
        </div>
    </div>
</div>

<!-- Bulk Actions -->
<div class="card mb-4">
    <div class="card-body">
        <form method="POST" action="?action=tags" id="bulkActionForm">
            <input type="hidden" name="bulk_action" id="bulkAction">
            
            <div class="row align-items-center">
                <div class="col-md-6">
                    <select class="form-select" id="bulkActionSelect" onchange="document.getElementById('bulkAction').value = this.value;">
                        <option value="">Bulk Actions</option>
                        <option value="delete">Delete Selected</option>
                    </select>
                </div>
                <div class="col-md-6">
                    <button type="submit" class="btn btn-outline-primary" onclick="return confirmBulkAction()">
                        Apply
                    </button>
                </div>
            </div>
        </form>
    </div>
</div>

<div class="card">
    <div class="card-header">
        <h5 class="mb-0">All Tags</h5>
    </div>
    <div class="card-body">
        <?php if (empty($tags)): ?>
        <div class="text-center py-4">
            <i class="fas fa-tags fa-3x text-muted mb-3"></i>
            <p class="text-muted">No tags found. Create your first tag!</p>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#tagModal">
                Create First Tag
            </button>
        </div>
        <?php else: ?>
        <div class="table-responsive">
            <table class="table table-striped table-hover">
                <thead>
                    <tr>
                        <th width="30">
                            <input type="checkbox" id="selectAll" onchange="toggleSelectAll(this)">
                        </th>
                        <th>Name</th>
                        <th>Slug</th>
                        <th>Description</th>
                        <th>Usage</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($tags as $tag): ?>
                    <tr>
                        <td>
                            <input type="checkbox" name="tag_ids[]" value="<?php echo $tag['id']; ?>" class="tag-checkbox">
                        </td>
                        <td>
                            <strong><?php echo htmlspecialchars($tag['name']); ?></strong>
                        </td>
                        <td>
                            <code><?php echo htmlspecialchars($tag['slug']); ?></code>
                        </td>
                        <td><?php echo htmlspecialchars($tag['description'] ?? '-'); ?></td>
                        <td>
                            <span class="badge bg-<?php 
                                $usage = $tag['usage_count'] ?? 0;
                                echo $usage > 10 ? 'success' : ($usage > 5 ? 'warning' : 'secondary');
                            ?>">
                                <?php echo $usage; ?> posts
                            </span>
                        </td>
                        <td>
                            <a href="?action=tags&edit=<?php echo $tag['id']; ?>" class="btn btn-sm btn-outline-primary">
                                <i class="fas fa-edit"></i>
                            </a>
                            <a href="?action=tags&delete=<?php echo $tag['id']; ?>" class="btn btn-sm btn-outline-danger" 
                               onclick="return confirm('Are you sure you want to delete this tag?')">
                                <i class="fas fa-trash"></i>
                            </a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Add/Edit Tag Modal -->
<div class="modal fade" id="tagModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><?php echo $edit_tag ? 'Edit Tag' : 'Add New Tag'; ?></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" action="?action=tags">
                <div class="modal-body">
                    <input type="hidden" name="action" value="<?php echo $edit_tag ? 'update_tag' : 'create_tag'; ?>">
                    <?php if ($edit_tag): ?>
                    <input type="hidden" name="id" value="<?php echo htmlspecialchars($edit_tag['id']); ?>">
                    <?php endif; ?>
                    
                    <div class="mb-3">
                        <label for="name" class="form-label">Tag Name *</label>
                        <input type="text" class="form-control" id="name" name="name" 
                               value="<?php echo htmlspecialchars($edit_tag['name'] ?? ''); ?>" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="slug" class="form-label">Slug *</label>
                        <input type="text" class="form-control" id="slug" name="slug" 
                               value="<?php echo htmlspecialchars($edit_tag['slug'] ?? ''); ?>" required>
                        <div class="form-text">URL-friendly version of the name</div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="description" class="form-label">Description</label>
                        <textarea class="form-control" id="description" name="description" 
                                  rows="3"><?php echo htmlspecialchars($edit_tag['description'] ?? ''); ?></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary"><?php echo $edit_tag ? 'Update Tag' : 'Create Tag'; ?></button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// Auto-generate slug from name
document.addEventListener('DOMContentLoaded', function() {
    const nameInput = document.getElementById('name');
    const slugInput = document.getElementById('slug');
    
    if (nameInput && slugInput) {
        nameInput.addEventListener('blur', function() {
            if (!slugInput.value) {
                const slug = this.value.toLowerCase()
                    .replace(/[^a-z0-9]+/g, '-')
                    .replace(/(^-|-$)+/g, '');
                slugInput.value = slug;
            }
        });
    }
});

// Bulk actions
function toggleSelectAll(source) {
    const checkboxes = document.querySelectorAll('.tag-checkbox');
    checkboxes.forEach(checkbox => {
        checkbox.checked = source.checked;
    });
}

function confirmBulkAction() {
    const action = document.getElementById('bulkAction').value;
    const checkedBoxes = document.querySelectorAll('.tag-checkbox:checked');
    
    if (checkedBoxes.length === 0) {
        alert('Please select at least one tag.');
        return false;
    }
    
    if (action === 'delete') {
        return confirm(`Are you sure you want to delete ${checkedBoxes.length} tag(s)?`);
    }
    
    return true;
}
</script>

<?php if ($_GET['edit'] ?? ''): ?>
<script>
document.addEventListener('DOMContentLoaded', function() {
    var tagModal = new bootstrap.Modal(document.getElementById('tagModal'));
    tagModal.show();
});
</script>
<?php endif; ?>