<?php
session_start();

// Redirect if already logged in
if (isset($_SESSION['user_id'])) {
    header('Location: /dashboard');
    exit;
}

// Process registration form
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $first_name = $_POST['first_name'] ?? '';
    $last_name = $_POST['last_name'] ?? '';
    $email = $_POST['email'] ?? '';
    $password = $_POST['password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    $terms = $_POST['terms'] ?? '';
    
    // Validation
    $errors = [];
    
    if (empty($first_name)) {
        $errors[] = 'First name is required';
    }
    
    if (empty($last_name)) {
        $errors[] = 'Last name is required';
    }
    
    if (empty($email)) {
        $errors[] = 'Email is required';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'Invalid email format';
    }
    
    if (empty($password)) {
        $errors[] = 'Password is required';
    } elseif (strlen($password) < 6) {
        $errors[] = 'Password must be at least 6 characters';
    }
    
    if ($password !== $confirm_password) {
        $errors[] = 'Passwords do not match';
    }
    
    if (!$terms) {
        $errors[] = 'You must agree to the terms and conditions';
    }
    
    if (empty($errors)) {
        // In real application, you would:
        // 1. Check if email already exists
        // 2. Hash password
        // 3. Create user in database
        // 4. Send verification email
        
        // For demo, auto-login and redirect
        $_SESSION['user_id'] = rand(1000, 9999);
        $_SESSION['user_email'] = $email;
        $_SESSION['user_name'] = $first_name . ' ' . $last_name;
        $_SESSION['user_level'] = 'new';
        
        header('Location: /placement-test');
        exit;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sign Up - Simturax English Learning</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .signup-section {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            padding: 40px 0;
        }
        .signup-card {
            background: white;
            border-radius: 20px;
            box-shadow: 0 25px 50px rgba(0,0,0,0.15);
            overflow: hidden;
        }
        .signup-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 40px;
            text-align: center;
        }
        .signup-body {
            padding: 40px;
        }
        .progress-steps {
            display: flex;
            justify-content: space-between;
            margin-bottom: 30px;
            position: relative;
        }
        .progress-steps::before {
            content: '';
            position: absolute;
            top: 15px;
            left: 0;
            right: 0;
            height: 3px;
            background: #e9ecef;
            z-index: 1;
        }
        .step {
            width: 30px;
            height: 30px;
            border-radius: 50%;
            background: white;
            border: 3px solid #e9ecef;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            font-size: 14px;
            position: relative;
            z-index: 2;
        }
        .step.active {
            border-color: #667eea;
            background: #667eea;
            color: white;
        }
        .step.completed {
            border-color: #28a745;
            background: #28a745;
            color: white;
        }
        .form-control {
            border-radius: 10px;
            padding: 12px 15px;
            border: 2px solid #e9ecef;
            transition: all 0.3s;
        }
        .form-control:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
        }
        .btn-signup {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
            border-radius: 10px;
            padding: 12px;
            font-weight: 600;
            transition: all 0.3s;
        }
        .btn-signup:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(102, 126, 234, 0.3);
        }
        .benefit-card {
            background: #f8f9ff;
            border-radius: 10px;
            padding: 20px;
            text-align: center;
            margin-bottom: 20px;
            transition: all 0.3s;
        }
        .benefit-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px rgba(102, 126, 234, 0.1);
        }
        .benefit-icon {
            width: 50px;
            height: 50px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 15px;
        }
        .password-strength {
            height: 5px;
            background: #e9ecef;
            border-radius: 5px;
            margin-top: 5px;
            overflow: hidden;
        }
        .password-strength-bar {
            height: 100%;
            width: 0%;
            transition: all 0.3s;
            border-radius: 5px;
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-light bg-white shadow-sm">
        <div class="container">
            <a class="navbar-brand fw-bold text-primary" href="/">
                <i class="fas fa-graduation-cap me-2"></i>Simturax
            </a>
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="/">Home</a>
                <a class="nav-link" href="/courses">Courses</a>
                <a class="nav-link" href="/placement-test">Test</a>
                <a class="nav-link" href="/login">Login</a>
                <a class="nav-link active" href="/signup">Sign Up</a>
            </div>
        </div>
    </nav>

    <!-- Sign Up Section -->
    <section class="signup-section">
        <div class="container">
            <div class="row justify-content-center">
                <div class="col-lg-10">
                    <div class="signup-card">
                        <div class="row g-0">
                            <!-- Registration Form -->
                            <div class="col-md-7">
                                <div class="signup-body">
                                    <h3 class="fw-bold mb-1">Start Your Journey</h3>
                                    <p class="text-muted mb-4">Create your account and discover your English level</p>
                                    
                                    <!-- Progress Steps -->
                                    <div class="progress-steps">
                                        <div class="step active">1</div>
                                        <div class="step">2</div>
                                        <div class="step">3</div>
                                    </div>
                                    
                                    <?php if (!empty($errors)): ?>
                                    <div class="alert alert-danger">
                                        <?php foreach ($errors as $error): ?>
                                        <div><?php echo htmlspecialchars($error); ?></div>
                                        <?php endforeach; ?>
                                    </div>
                                    <?php endif; ?>

                                    <form method="POST" action="" id="signupForm">
                                        <div class="row">
                                            <div class="col-md-6">
                                                <div class="mb-3">
                                                    <label for="first_name" class="form-label">First Name</label>
                                                    <input type="text" class="form-control" id="first_name" name="first_name" 
                                                           value="<?php echo htmlspecialchars($_POST['first_name'] ?? ''); ?>" 
                                                           placeholder="Enter your first name" required>
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <div class="mb-3">
                                                    <label for="last_name" class="form-label">Last Name</label>
                                                    <input type="text" class="form-control" id="last_name" name="last_name" 
                                                           value="<?php echo htmlspecialchars($_POST['last_name'] ?? ''); ?>" 
                                                           placeholder="Enter your last name" required>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label for="email" class="form-label">Email Address</label>
                                            <input type="email" class="form-control" id="email" name="email" 
                                                   value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>" 
                                                   placeholder="Enter your email" required>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label for="password" class="form-label">Password</label>
                                            <input type="password" class="form-control" id="password" name="password" 
                                                   placeholder="Create a password" required minlength="6">
                                            <div class="password-strength">
                                                <div class="password-strength-bar" id="passwordStrengthBar"></div>
                                            </div>
                                            <small class="text-muted">Must be at least 6 characters</small>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label for="confirm_password" class="form-label">Confirm Password</label>
                                            <input type="password" class="form-control" id="confirm_password" name="confirm_password" 
                                                   placeholder="Confirm your password" required>
                                        </div>
                                        
                                        <div class="mb-4">
                                            <div class="form-check">
                                                <input class="form-check-input" type="checkbox" id="terms" name="terms" 
                                                       <?php echo isset($_POST['terms']) ? 'checked' : ''; ?>>
                                                <label class="form-check-label" for="terms">
                                                    I agree to the <a href="/terms" class="text-decoration-none">Terms of Service</a> 
                                                    and <a href="/privacy" class="text-decoration-none">Privacy Policy</a>
                                                </label>
                                            </div>
                                        </div>
                                        
                                        <button type="submit" class="btn btn-signup text-white w-100 mb-3">
                                            Create Account & Start Learning
                                        </button>
                                        
                                        <div class="text-center">
                                            <span class="text-muted">Already have an account? </span>
                                            <a href="/login" class="text-decoration-none fw-bold">Sign in here</a>
                                        </div>
                                    </form>
                                </div>
                            </div>
                            
                            <!-- Benefits Side -->
                            <div class="col-md-5 bg-light">
                                <div class="signup-body h-100 d-flex flex-column justify-content-center">
                                    <h4 class="fw-bold mb-4 text-center">Join 50,000+ Learners</h4>
                                    
                                    <div class="benefit-card">
                                        <div class="benefit-icon">
                                            <i class="fas fa-bolt text-white"></i>
                                        </div>
                                        <h6>Free Placement Test</h6>
                                        <p class="text-muted small mb-0">Discover your exact English level in 15 minutes</p>
                                    </div>
                                    
                                    <div class="benefit-card">
                                        <div class="benefit-icon">
                                            <i class="fas fa-route text-white"></i>
                                        </div>
                                        <h6>Personalized Path</h6>
                                        <p class="text-muted small mb-0">Get a custom learning plan tailored to your goals</p>
                                    </div>
                                    
                                    <div class="benefit-card">
                                        <div class="benefit-icon">
                                            <i class="fas fa-chart-line text-white"></i>
                                        </div>
                                        <h6>Track Progress</h6>
                                        <p class="text-muted small mb-0">Monitor your improvement with detailed analytics</p>
                                    </div>
                                    
                                    <div class="benefit-card">
                                        <div class="benefit-icon">
                                            <i class="fas fa-award text-white"></i>
                                        </div>
                                        <h6>Get Certified</h6>
                                        <p class="text-muted small mb-0">Earn certificates as you progress through levels</p>
                                    </div>
                                    
                                    <div class="mt-4 p-3 bg-white rounded text-center">
                                        <div class="row">
                                            <div class="col-4">
                                                <h5 class="fw-bold text-primary mb-1">50K+</h5>
                                                <small class="text-muted">Students</small>
                                            </div>
                                            <div class="col-4">
                                                <h5 class="fw-bold text-success mb-1">94%</h5>
                                                <small class="text-muted">Success Rate</small>
                                            </div>
                                            <div class="col-4">
                                                <h5 class="fw-bold text-info mb-1">A1-C2</h5>
                                                <small class="text-muted">All Levels</small>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Password strength indicator
        const passwordInput = document.getElementById('password');
        const strengthBar = document.getElementById('passwordStrengthBar');
        
        passwordInput.addEventListener('input', function() {
            const password = this.value;
            let strength = 0;
            
            if (password.length >= 6) strength += 25;
            if (password.match(/[a-z]/) && password.match(/[A-Z]/)) strength += 25;
            if (password.match(/\d/)) strength += 25;
            if (password.match(/[^a-zA-Z\d]/)) strength += 25;
            
            strengthBar.style.width = strength + '%';
            
            if (strength < 50) {
                strengthBar.style.background = '#dc3545';
            } else if (strength < 75) {
                strengthBar.style.background = '#ffc107';
            } else {
                strengthBar.style.background = '#28a745';
            }
        });
        
        // Form validation
        const form = document.getElementById('signupForm');
        const confirmPassword = document.getElementById('confirm_password');
        
        form.addEventListener('submit', function(e) {
            const password = passwordInput.value;
            const confirm = confirmPassword.value;
            
            if (password !== confirm) {
                e.preventDefault();
                confirmPassword.setCustomValidity('Passwords do not match');
                confirmPassword.reportValidity();
            } else {
                confirmPassword.setCustomValidity('');
            }
        });
        
        confirmPassword.addEventListener('input', function() {
            if (this.value !== passwordInput.value) {
                this.setCustomValidity('Passwords do not match');
            } else {
                this.setCustomValidity('');
            }
        });
    </script>
</body>
</html>