<?php
session_start();
if (!isset($_SESSION['admin_logged_in'])) {
    die('Please login first');
}

error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../app/config/database.php';
$database = new Database();
$db = $database->getConnection();

if (!$db) {
    die('Database connection failed');
}

echo "<!DOCTYPE html>
<html>
<head>
    <title>Database Setup - Simturax</title>
    <link href='https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css' rel='stylesheet'>
    <link href='https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css' rel='stylesheet'>
    <style>
        body {
            background: #f8f9fa;
        }
        .container {
            max-width: 900px;
        }
        .card {
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            border: none;
            border-radius: 10px;
        }
        .progress {
            height: 8px;
        }
    </style>
</head>
<body>
    <div class='container mt-5'>
        <div class='text-center mb-4'>
            <h1><i class='fas fa-database text-primary me-2'></i>Database Setup</h1>
            <p class='text-muted'>Setting up database tables for Simturax English Learning Platform</p>
        </div>
        
        <div class='card'>
            <div class='card-header bg-primary text-white'>
                <h4 class='mb-0'><i class='fas fa-cogs me-2'></i>Database Tables Creation</h4>
            </div>
            <div class='card-body'>";

$tables = [
    // Users table
    "CREATE TABLE IF NOT EXISTS users (
        id INT AUTO_INCREMENT PRIMARY KEY,
        email VARCHAR(255) UNIQUE NOT NULL,
        password VARCHAR(255) NOT NULL,
        first_name VARCHAR(100),
        last_name VARCHAR(100),
        current_level ENUM('A1', 'A2', 'B1', 'B2', 'C1', 'C2'),
        test_score INT,
        learning_path JSON,
        status ENUM('active', 'inactive', 'suspended') DEFAULT 'active',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )",
    
    // Posts table
    "CREATE TABLE IF NOT EXISTS posts (
        id INT AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255) NOT NULL,
        content TEXT NOT NULL,
        category_id INT NULL,
        status ENUM('draft', 'published') DEFAULT 'draft',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )",
    
    // Pages table
    "CREATE TABLE IF NOT EXISTS pages (
        id INT AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255) NOT NULL,
        slug VARCHAR(255) UNIQUE NOT NULL,
        content TEXT NOT NULL,
        status ENUM('draft', 'published') DEFAULT 'draft',
        meta_title VARCHAR(255) NULL,
        meta_description TEXT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )",
    
    // Categories table
    "CREATE TABLE IF NOT EXISTS categories (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL,
        description TEXT NULL,
        type ENUM('post', 'course', 'general') DEFAULT 'post',
        status ENUM('active', 'inactive') DEFAULT 'active',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )",
    
    // Courses table
    "CREATE TABLE IF NOT EXISTS courses (
        id INT AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255) NOT NULL,
        description TEXT,
        level ENUM('A1', 'A2', 'B1', 'B2', 'C1', 'C2') DEFAULT 'A1',
        skill_type ENUM('reading', 'writing', 'speaking', 'grammar', 'vocabulary') DEFAULT 'reading',
        status ENUM('draft', 'published') DEFAULT 'draft',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )",
    
    // Media table
    "CREATE TABLE IF NOT EXISTS media (
        id INT AUTO_INCREMENT PRIMARY KEY,
        filename VARCHAR(255) NOT NULL,
        original_name VARCHAR(255) NOT NULL,
        file_path VARCHAR(500) NOT NULL,
        file_size INT NOT NULL,
        mime_type VARCHAR(100) NOT NULL,
        uploaded_by VARCHAR(100) NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )",
    
    // Settings table
    "CREATE TABLE IF NOT EXISTS settings (
        id INT AUTO_INCREMENT PRIMARY KEY,
        setting_key VARCHAR(100) UNIQUE NOT NULL,
        setting_value TEXT NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )",
    
    // Ads table
    "CREATE TABLE IF NOT EXISTS ads (
        id INT AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255) NOT NULL,
        ad_type ENUM('banner', 'sidebar', 'popup', 'inline', 'video') DEFAULT 'banner',
        ad_code TEXT NOT NULL,
        position VARCHAR(50) NOT NULL,
        status ENUM('active', 'inactive') DEFAULT 'active',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )",
    
    // AI settings table
    "CREATE TABLE IF NOT EXISTS ai_settings (
        id INT AUTO_INCREMENT PRIMARY KEY,
        setting_key VARCHAR(100) UNIQUE NOT NULL,
        setting_value TEXT NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )",
    
    // Payments table
    "CREATE TABLE IF NOT EXISTS payments (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        amount DECIMAL(10,2) NOT NULL,
        currency VARCHAR(3) DEFAULT 'USD',
        plan_name VARCHAR(100) NOT NULL,
        status ENUM('pending', 'completed', 'failed', 'refunded', 'cancelled') DEFAULT 'pending',
        refund_amount DECIMAL(10,2) NULL,
        refund_reason TEXT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )",
    
    // Subscription plans table
    "CREATE TABLE IF NOT EXISTS subscription_plans (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL,
        price DECIMAL(10,2) NOT NULL,
        duration_days INT NOT NULL,
        features TEXT NULL,
        status ENUM('active', 'inactive') DEFAULT 'active',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )",
    
    // Languages table
    "CREATE TABLE IF NOT EXISTS languages (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL,
        code VARCHAR(5) UNIQUE NOT NULL,
        locale VARCHAR(10) NULL,
        status ENUM('active', 'inactive') DEFAULT 'active',
        is_default BOOLEAN DEFAULT FALSE,
        translation_progress INT DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )",
    
    // Notifications table
    "CREATE TABLE IF NOT EXISTS notifications (
        id INT AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255) NOT NULL,
        message TEXT NOT NULL,
        type ENUM('info', 'success', 'warning', 'error', 'promotion') DEFAULT 'info',
        audience ENUM('all', 'students', 'teachers', 'inactive', 'premium') DEFAULT 'all',
        schedule ENUM('immediate', 'scheduled') DEFAULT 'immediate',
        scheduled_date DATETIME NULL,
        status ENUM('pending', 'sent', 'cancelled') DEFAULT 'pending',
        sent_at DATETIME NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )",
    
    // Tags table
    "CREATE TABLE IF NOT EXISTS tags (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(100) NOT NULL,
        slug VARCHAR(100) UNIQUE NOT NULL,
        description TEXT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )",
    
    // Post tags relationship table
    "CREATE TABLE IF NOT EXISTS post_tags (
        id INT AUTO_INCREMENT PRIMARY KEY,
        post_id INT NOT NULL,
        tag_id INT NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        UNIQUE KEY unique_post_tag (post_id, tag_id)
    )",
    
    // Placement test questions table
    "CREATE TABLE IF NOT EXISTS placement_questions (
        id INT AUTO_INCREMENT PRIMARY KEY,
        question_text TEXT NOT NULL,
        options JSON NOT NULL,
        correct_answer VARCHAR(10),
        difficulty_level ENUM('A1', 'A2', 'B1', 'B2', 'C1', 'C2'),
        question_type ENUM('grammar', 'vocabulary', 'reading', 'listening'),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )",
    
    // Learning content table
    "CREATE TABLE IF NOT EXISTS learning_content (
        id INT AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255) NOT NULL,
        content TEXT,
        level ENUM('A1', 'A2', 'B1', 'B2', 'C1', 'C2'),
        skill_type ENUM('reading', 'speaking', 'writing', 'grammar', 'vocabulary'),
        order_index INT,
        is_active BOOLEAN DEFAULT TRUE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )",
    
    // User progress tracking table
    "CREATE TABLE IF NOT EXISTS user_progress (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT,
        content_id INT,
        completed BOOLEAN DEFAULT FALSE,
        score DECIMAL(5,2),
        time_spent INT,
        completed_at TIMESTAMP,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id),
        FOREIGN KEY (content_id) REFERENCES learning_content(id)
    )"
];

$total_tables = count($tables);
$created_count = 0;
$error_count = 0;

echo "<div class='progress mb-4'>
        <div class='progress-bar progress-bar-striped progress-bar-animated' role='progressbar' style='width: 0%' id='progressBar'></div>
      </div>";

echo "<div class='table-responsive'>
        <table class='table table-striped'>
            <thead>
                <tr>
                    <th>Table Name</th>
                    <th>Status</th>
                    <th>Message</th>
                </tr>
            </thead>
            <tbody>";

foreach ($tables as $table) {
    $table_name = preg_match('/CREATE TABLE IF NOT EXISTS (\w+)/', $table, $matches) ? $matches[1] : 'Unknown';
    
    try {
        $db->exec($table);
        $status = 'success';
        $message = 'Created successfully';
        $created_count++;
        echo "<tr>
                <td><strong>{$table_name}</strong></td>
                <td><span class='badge bg-success'><i class='fas fa-check me-1'></i>Success</span></td>
                <td class='text-success'>{$message}</td>
              </tr>";
    } catch (Exception $e) {
        $status = 'warning';
        $message = $e->getMessage();
        $error_count++;
        echo "<tr>
                <td><strong>{$table_name}</strong></td>
                <td><span class='badge bg-warning'><i class='fas fa-exclamation-triangle me-1'></i>Warning</span></td>
                <td class='text-warning'>{$message}</td>
              </tr>";
    }
    
    // Update progress bar
    $progress = (($created_count + $error_count) / $total_tables) * 100;
    echo "<script>document.getElementById('progressBar').style.width = '{$progress}%';</script>";
    flush();
    ob_flush();
    usleep(100000); // Small delay for visual effect
}

echo "</tbody></table></div>";

// Add sample data
echo "<div class='mt-4'>
        <h5><i class='fas fa-database me-2'></i>Adding Sample Data</h5>
        <div class='row'>";

try {
    // Sample categories
    $check = $db->query("SELECT COUNT(*) as count FROM categories")->fetch();
    if ($check['count'] == 0) {
        $db->exec("INSERT INTO categories (name, type, description) VALUES 
                  ('Grammar', 'post', 'Grammar related content'),
                  ('Vocabulary', 'post', 'Vocabulary building exercises'),
                  ('Speaking', 'post', 'Speaking practice materials'),
                  ('Writing', 'post', 'Writing skills development'),
                  ('Beginner Courses', 'course', 'Courses for A1-A2 levels'),
                  ('Intermediate Courses', 'course', 'Courses for B1-B2 levels'),
                  ('Advanced Courses', 'course', 'Courses for C1-C2 levels')");
        echo "<div class='col-md-6 mb-2'>
                <div class='alert alert-success'>
                    <i class='fas fa-check me-2'></i>Sample categories added
                </div>
              </div>";
    }
} catch (Exception $e) {
    echo "<div class='col-md-6 mb-2'>
            <div class='alert alert-warning'>
                <i class='fas fa-exclamation-triangle me-2'></i>Categories: " . $e->getMessage() . "
            </div>
          </div>";
}

try {
    // Sample posts
    $check = $db->query("SELECT COUNT(*) as count FROM posts")->fetch();
    if ($check['count'] == 0) {
        $db->exec("INSERT INTO posts (title, content, status) VALUES 
                  ('Welcome to Simturax English Learning', 'This is your first post. Welcome to our English learning platform!', 'published'),
                  ('Tips for Improving Your Vocabulary', 'Learn effective strategies to expand your English vocabulary daily.', 'published'),
                  ('Common Grammar Mistakes to Avoid', 'Discover the most common grammar errors and how to fix them.', 'draft')");
        echo "<div class='col-md-6 mb-2'>
                <div class='alert alert-success'>
                    <i class='fas fa-check me-2'></i>Sample posts added
                </div>
              </div>";
    }
} catch (Exception $e) {
    echo "<div class='col-md-6 mb-2'>
            <div class='alert alert-warning'>
                <i class='fas fa-exclamation-triangle me-2'></i>Posts: " . $e->getMessage() . "
            </div>
          </div>";
}

try {
    // Sample courses
    $check = $db->query("SELECT COUNT(*) as count FROM courses")->fetch();
    if ($check['count'] == 0) {
        $db->exec("INSERT INTO courses (title, description, level, skill_type, status) VALUES 
                  ('Basic English Grammar', 'Learn fundamental grammar rules and structures', 'A1', 'grammar', 'published'),
                  ('Everyday Vocabulary Builder', 'Expand your daily use vocabulary', 'A2', 'vocabulary', 'published'),
                  ('Conversation Practice', 'Improve your speaking skills through dialogues', 'B1', 'speaking', 'published'),
                  ('Business Writing', 'Professional writing skills for the workplace', 'B2', 'writing', 'draft')");
        echo "<div class='col-md-6 mb-2'>
                <div class='alert alert-success'>
                    <i class='fas fa-check me-2'></i>Sample courses added
                </div>
              </div>";
    }
} catch (Exception $e) {
    echo "<div class='col-md-6 mb-2'>
            <div class='alert alert-warning'>
                <i class='fas fa-exclamation-triangle me-2'></i>Courses: " . $e->getMessage() . "
            </div>
          </div>";
}

try {
    // Sample settings
    $default_settings = [
        'site_title' => 'Simturax English Learning',
        'site_description' => 'Learn English online with personalized courses and placement tests',
        'admin_email' => 'admin@simturax.com',
        'items_per_page' => '10',
        'maintenance_mode' => '0',
        'registration_enabled' => '1',
        'default_language' => 'en',
        'contact_email' => 'contact@simturax.com'
    ];
    
    foreach ($default_settings as $key => $value) {
        $check = $db->query("SELECT COUNT(*) as count FROM settings WHERE setting_key = '$key'")->fetch();
        if ($check['count'] == 0) {
            $db->exec("INSERT INTO settings (setting_key, setting_value) VALUES ('$key', '$value')");
        }
    }
    echo "<div class='col-md-6 mb-2'>
            <div class='alert alert-success'>
                <i class='fas fa-check me-2'></i>Default settings added
            </div>
          </div>";
} catch (Exception $e) {
    echo "<div class='col-md-6 mb-2'>
            <div class='alert alert-warning'>
                <i class='fas fa-exclamation-triangle me-2'></i>Settings: " . $e->getMessage() . "
            </div>
          </div>";
}

try {
    // Sample languages
    $check = $db->query("SELECT COUNT(*) as count FROM languages")->fetch();
    if ($check['count'] == 0) {
        $db->exec("INSERT INTO languages (name, code, locale, status, is_default, translation_progress) VALUES 
                  ('English', 'en', 'en_US', 'active', 1, 100),
                  ('Spanish', 'es', 'es_ES', 'active', 0, 75),
                  ('French', 'fr', 'fr_FR', 'active', 0, 60),
                  ('German', 'de', 'de_DE', 'inactive', 0, 45)");
        echo "<div class='col-md-6 mb-2'>
                <div class='alert alert-success'>
                    <i class='fas fa-check me-2'></i>Sample languages added
                </div>
              </div>";
    }
} catch (Exception $e) {
    echo "<div class='col-md-6 mb-2'>
            <div class='alert alert-warning'>
                <i class='fas fa-exclamation-triangle me-2'></i>Languages: " . $e->getMessage() . "
            </div>
          </div>";
}

try {
    // Sample tags
    $check = $db->query("SELECT COUNT(*) as count FROM tags")->fetch();
    if ($check['count'] == 0) {
        $db->exec("INSERT INTO tags (name, slug, description) VALUES 
                  ('Grammar', 'grammar', 'Grammar related content'),
                  ('Vocabulary', 'vocabulary', 'Vocabulary building'),
                  ('Speaking', 'speaking', 'Speaking practice'),
                  ('Writing', 'writing', 'Writing skills'),
                  ('Listening', 'listening', 'Listening comprehension'),
                  ('Beginner', 'beginner', 'A1-A2 level content'),
                  ('Intermediate', 'intermediate', 'B1-B2 level content'),
                  ('Advanced', 'advanced', 'C1-C2 level content')");
        echo "<div class='col-md-6 mb-2'>
                <div class='alert alert-success'>
                    <i class='fas fa-check me-2'></i>Sample tags added
                </div>
              </div>";
    }
} catch (Exception $e) {
    echo "<div class='col-md-6 mb-2'>
            <div class='alert alert-warning'>
                <i class='fas fa-exclamation-triangle me-2'></i>Tags: " . $e->getMessage() . "
            </div>
          </div>";
}

try {
    // Sample subscription plans
    $check = $db->query("SELECT COUNT(*) as count FROM subscription_plans")->fetch();
    if ($check['count'] == 0) {
        $db->exec("INSERT INTO subscription_plans (name, price, duration_days, features, status) VALUES 
                  ('Basic Monthly', 9.99, 30, 'Basic access, Limited courses', 'active'),
                  ('Premium Monthly', 29.99, 30, 'Full access, All courses, Progress tracking', 'active'),
                  ('Premium Annual', 299.99, 365, 'Full access, All courses, Progress tracking, Priority support', 'active')");
        echo "<div class='col-md-6 mb-2'>
                <div class='alert alert-success'>
                    <i class='fas fa-check me-2'></i>Sample subscription plans added
                </div>
              </div>";
    }
} catch (Exception $e) {
    echo "<div class='col-md-6 mb-2'>
            <div class='alert alert-warning'>
                <i class='fas fa-exclamation-triangle me-2'></i>Subscription plans: " . $e->getMessage() . "
            </div>
          </div>";
}

echo "</div></div>";

// Summary
$success_rate = round(($created_count / $total_tables) * 100, 2);

echo "<div class='mt-4 p-4 rounded' style='background: #e8f5e8; border-left: 4px solid #28a745;'>
        <h4><i class='fas fa-check-circle text-success me-2'></i>Setup Completed!</h4>
        <div class='row mt-3'>
            <div class='col-md-3 text-center'>
                <h2 class='text-success'>{$created_count}</h2>
                <p class='mb-0'>Tables Created</p>
            </div>
            <div class='col-md-3 text-center'>
                <h2 class='text-warning'>{$error_count}</h2>
                <p class='mb-0'>Warnings</p>
            </div>
            <div class='col-md-3 text-center'>
                <h2 class='text-primary'>{$total_tables}</h2>
                <p class='mb-0'>Total Tables</p>
            </div>
            <div class='col-md-3 text-center'>
                <h2 class='text-info'>{$success_rate}%</h2>
                <p class='mb-0'>Success Rate</p>
            </div>
        </div>
        
        <div class='mt-4'>
            <div class='alert alert-info'>
                <h6><i class='fas fa-info-circle me-2'></i>Next Steps:</h6>
                <ul class='mb-0'>
                    <li>Visit the admin panel to configure your settings</li>
                    <li>Add your first courses and learning content</li>
                    <li>Configure payment and notification settings</li>
                    <li>Test the user registration and placement test</li>
                </ul>
            </div>
        </div>
        
        <div class='text-center mt-4'>
            <a href='index.php' class='btn btn-success btn-lg me-3'>
                <i class='fas fa-rocket me-2'></i>Launch Admin Panel
            </a>
            <a href='../' class='btn btn-outline-primary btn-lg'>
                <i class='fas fa-eye me-2'></i>View Website
            </a>
        </div>
      </div>";

echo "</div></div></div></body></html>";
?>