<?php
// Display courses - no form processing here
if (!isset($db) || !$db) {
    echo '<div class="alert alert-danger">Database connection not available</div>';
    return;
}

try {
    // Fetch courses (read-only operations)
    $query = "SELECT * FROM courses ORDER BY level, skill_type, title";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $courses = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Get course for editing (read-only)
    $edit_course = null;
    if ($_GET['edit'] ?? '') {
        $query = "SELECT * FROM courses WHERE id = ?";
        $stmt = $db->prepare($query);
        $stmt->execute([$_GET['edit']]);
        $edit_course = $stmt->fetch(PDO::FETCH_ASSOC);
    }
} catch (Exception $e) {
    echo '<div class="alert alert-danger">Database error: ' . htmlspecialchars($e->getMessage()) . '</div>';
    $courses = [];
}

// Skill type names
$skillTypes = [
    'reading' => 'Reading Skills',
    'writing' => 'Writing Skills',
    'speaking' => 'Speaking Skills',
    'grammar' => 'Grammar',
    'vocabulary' => 'Vocabulary'
];

// Level names
$levelNames = [
    'A1' => 'Beginner',
    'A2' => 'Elementary', 
    'B1' => 'Intermediate',
    'B2' => 'Upper Intermediate',
    'C1' => 'Advanced',
    'C2' => 'Proficient'
];
?>
<div class="row">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h2>Courses Manager</h2>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#courseModal">
                <i class="fas fa-plus me-2"></i>Add New Course
            </button>
        </div>
    </div>
</div>

<div class="card">
    <div class="card-header">
        <h5 class="mb-0">All Courses</h5>
    </div>
    <div class="card-body">
        <?php if (empty($courses)): ?>
        <div class="text-center py-4">
            <i class="fas fa-book fa-3x text-muted mb-3"></i>
            <p class="text-muted">No courses found. Create your first course!</p>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#courseModal">
                Create First Course
            </button>
        </div>
        <?php else: ?>
        <div class="table-responsive">
            <table class="table table-striped table-hover">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Title</th>
                        <th>Skill</th>
                        <th>Level</th>
                        <th>Status</th>
                        <th>Created</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($courses as $course): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($course['id']); ?></td>
                        <td>
                            <strong><?php echo htmlspecialchars($course['title']); ?></strong>
                            <?php if ($course['status'] == 'draft'): ?>
                                <span class="badge bg-warning ms-2">Draft</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <span class="badge bg-primary"><?php echo $skillTypes[$course['skill_type']] ?? $course['skill_type']; ?></span>
                        </td>
                        <td>
                            <span class="badge bg-info"><?php echo $course['level'] . ' - ' . ($levelNames[$course['level']] ?? 'Unknown'); ?></span>
                        </td>
                        <td>
                            <span class="badge bg-<?php echo $course['status'] == 'published' ? 'success' : 'warning'; ?>">
                                <?php echo ucfirst($course['status']); ?>
                            </span>
                        </td>
                        <td><?php echo date('M j, Y', strtotime($course['created_at'])); ?></td>
                        <td>
                            <a href="?action=courses&edit=<?php echo $course['id']; ?>" class="btn btn-sm btn-outline-primary">
                                <i class="fas fa-edit"></i>
                            </a>
                            <a href="?action=courses&delete=<?php echo $course['id']; ?>" class="btn btn-sm btn-outline-danger" 
                               onclick="return confirm('Are you sure you want to delete this course?')">
                                <i class="fas fa-trash"></i>
                            </a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Add/Edit Course Modal -->
<div class="modal fade" id="courseModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><?php echo $edit_course ? 'Edit Course' : 'Add New Course'; ?></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" action="?action=courses">
                <div class="modal-body">
                    <input type="hidden" name="action" value="<?php echo $edit_course ? 'update_course' : 'create_course'; ?>">
                    <?php if ($edit_course): ?>
                    <input type="hidden" name="id" value="<?php echo htmlspecialchars($edit_course['id']); ?>">
                    <?php endif; ?>
                    
                    <div class="mb-3">
                        <label for="title" class="form-label">Course Title *</label>
                        <input type="text" class="form-control" id="title" name="title" 
                               value="<?php echo htmlspecialchars($edit_course['title'] ?? ''); ?>" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="description" class="form-label">Description</label>
                        <textarea class="form-control" id="description" name="description" 
                                  rows="4"><?php echo htmlspecialchars($edit_course['description'] ?? ''); ?></textarea>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="skill_type" class="form-label">Skill Type *</label>
                                <select class="form-select" id="skill_type" name="skill_type" required>
                                    <?php foreach ($skillTypes as $key => $name): ?>
                                    <option value="<?php echo $key; ?>" 
                                        <?php echo ($edit_course['skill_type'] ?? '') == $key ? 'selected' : ''; ?>>
                                        <?php echo $name; ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="level" class="form-label">Level *</label>
                                <select class="form-select" id="level" name="level" required>
                                    <?php foreach ($levelNames as $key => $name): ?>
                                    <option value="<?php echo $key; ?>" 
                                        <?php echo ($edit_course['level'] ?? '') == $key ? 'selected' : ''; ?>>
                                        <?php echo $key . ' - ' . $name; ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="status" class="form-label">Status *</label>
                        <select class="form-select" id="status" name="status" required>
                            <option value="draft" <?php echo ($edit_course['status'] ?? '') == 'draft' ? 'selected' : ''; ?>>Draft</option>
                            <option value="published" <?php echo ($edit_course['status'] ?? '') == 'published' ? 'selected' : ''; ?>>Published</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary"><?php echo $edit_course ? 'Update Course' : 'Create Course'; ?></button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php if ($_GET['edit'] ?? ''): ?>
<script>
document.addEventListener('DOMContentLoaded', function() {
    var courseModal = new bootstrap.Modal(document.getElementById('courseModal'));
    courseModal.show();
});
</script>
<?php endif; ?>