<?php
// Display language manager - no form processing here
if (!isset($db) || !$db) {
    echo '<div class="alert alert-danger">Database connection not available</div>';
    return;
}

try {
    // Fetch languages
    $query = "SELECT * FROM languages ORDER BY is_default DESC, name";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $languages = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Get default language from settings
    $query = "SELECT setting_value FROM settings WHERE setting_key = 'default_language'";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $default_language = $stmt->fetch(PDO::FETCH_ASSOC);
    $default_language_code = $default_language['setting_value'] ?? 'en';

    // Get language for editing
    $edit_language = null;
    if ($_GET['edit'] ?? '') {
        $query = "SELECT * FROM languages WHERE id = ?";
        $stmt = $db->prepare($query);
        $stmt->execute([$_GET['edit']]);
        $edit_language = $stmt->fetch(PDO::FETCH_ASSOC);
    }
} catch (Exception $e) {
    echo '<div class="alert alert-warning">Language database not configured yet</div>';
    $languages = [];
    $default_language_code = 'en';
}

// Default languages if none in database
if (empty($languages)) {
    $languages = [
        [
            'id' => '1',
            'name' => 'English',
            'code' => 'en',
            'locale' => 'en_US',
            'status' => 'active',
            'is_default' => '1',
            'translation_progress' => '100'
        ],
        [
            'id' => '2', 
            'name' => 'Spanish',
            'code' => 'es',
            'locale' => 'es_ES',
            'status' => 'active',
            'is_default' => '0',
            'translation_progress' => '75'
        ],
        [
            'id' => '3',
            'name' => 'French',
            'code' => 'fr', 
            'locale' => 'fr_FR',
            'status' => 'active',
            'is_default' => '0',
            'translation_progress' => '60'
        ],
        [
            'id' => '4',
            'name' => 'German',
            'code' => 'de',
            'locale' => 'de_DE',
            'status' => 'inactive',
            'is_default' => '0',
            'translation_progress' => '45'
        ]
    ];
}

// Common languages for quick add
$common_languages = [
    'Arabic' => ['code' => 'ar', 'locale' => 'ar_SA'],
    'Chinese' => ['code' => 'zh', 'locale' => 'zh_CN'],
    'Hindi' => ['code' => 'hi', 'locale' => 'hi_IN'],
    'Portuguese' => ['code' => 'pt', 'locale' => 'pt_BR'],
    'Russian' => ['code' => 'ru', 'locale' => 'ru_RU'],
    'Japanese' => ['code' => 'ja', 'locale' => 'ja_JP'],
    'Italian' => ['code' => 'it', 'locale' => 'it_IT'],
    'Korean' => ['code' => 'ko', 'locale' => 'ko_KR'],
    'Turkish' => ['code' => 'tr', 'locale' => 'tr_TR']
];
?>
<div class="row">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h2>Language Manager</h2>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#languageModal">
                <i class="fas fa-plus me-2"></i>Add Language
            </button>
        </div>
    </div>
</div>

<div class="row">
    <div class="col-md-8">
        <!-- Installed Languages -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0">Installed Languages</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead>
                            <tr>
                                <th>Language</th>
                                <th>Code</th>
                                <th>Status</th>
                                <th>Translation</th>
                                <th>Default</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($languages as $language): ?>
                            <tr>
                                <td>
                                    <strong><?php echo htmlspecialchars($language['name']); ?></strong>
                                    <?php if ($language['code'] == $default_language_code || $language['is_default']): ?>
                                        <span class="badge bg-primary ms-2">Default</span>
                                    <?php endif; ?>
                                </td>
                                <td><code><?php echo htmlspecialchars($language['code']); ?></code></td>
                                <td>
                                    <span class="badge bg-<?php echo $language['status'] == 'active' ? 'success' : 'secondary'; ?>">
                                        <?php echo ucfirst($language['status']); ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="progress" style="height: 6px;">
                                        <div class="progress-bar" role="progressbar" 
                                             style="width: <?php echo $language['translation_progress'] ?? '0'; ?>%">
                                        </div>
                                    </div>
                                    <small class="text-muted"><?php echo $language['translation_progress'] ?? '0'; ?>%</small>
                                </td>
                                <td>
                                    <?php if ($language['code'] != $default_language_code && !$language['is_default']): ?>
                                    <a href="?action=language&set_default=<?php echo $language['code']; ?>" 
                                       class="btn btn-sm btn-outline-primary"
                                       onclick="return confirm('Set <?php echo htmlspecialchars($language['name']); ?> as default language?')">
                                        Set Default
                                    </a>
                                    <?php else: ?>
                                    <span class="badge bg-primary">Default</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <a href="?action=language&edit=<?php echo $language['id']; ?>" 
                                       class="btn btn-sm btn-outline-primary">
                                        <i class="fas fa-edit"></i>
                                    </a>
                                    <?php if ($language['code'] != 'en'): ?>
                                    <a href="?action=language&delete=<?php echo $language['id']; ?>" 
                                       class="btn btn-sm btn-outline-danger"
                                       onclick="return confirm('Are you sure you want to delete <?php echo htmlspecialchars($language['name']); ?>?')">
                                        <i class="fas fa-trash"></i>
                                    </a>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        
        <!-- Translation Progress -->
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Translation Progress</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <?php foreach ($languages as $language): ?>
                    <?php if ($language['code'] != 'en'): ?>
                    <div class="col-md-6 mb-3">
                        <div class="card">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center mb-2">
                                    <h6 class="mb-0"><?php echo htmlspecialchars($language['name']); ?></h6>
                                    <span class="badge bg-<?php 
                                        $progress = $language['translation_progress'] ?? 0;
                                        echo $progress >= 90 ? 'success' : ($progress >= 50 ? 'warning' : 'danger');
                                    ?>">
                                        <?php echo $progress; ?>%
                                    </span>
                                </div>
                                <div class="progress mb-2" style="height: 8px;">
                                    <div class="progress-bar" role="progressbar" 
                                         style="width: <?php echo $progress; ?>%">
                                    </div>
                                </div>
                                <div class="btn-group w-100">
                                    <button class="btn btn-sm btn-outline-primary" disabled>
                                        <i class="fas fa-pencil-alt me-1"></i>Translate
                                    </button>
                                    <button class="btn btn-sm btn-outline-info" disabled>
                                        <i class="fas fa-download me-1"></i>Export
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-md-4">
        <!-- Quick Actions -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0">Quick Actions</h5>
            </div>
            <div class="card-body">
                <div class="d-grid gap-2">
                    <button class="btn btn-outline-primary" disabled>
                        <i class="fas fa-sync-alt me-2"></i>Scan for New Strings
                    </button>
                    <button class="btn btn-outline-success" disabled>
                        <i class="fas fa-download me-2"></i>Export All Translations
                    </button>
                    <button class="btn btn-outline-info" disabled>
                        <i class="fas fa-upload me-2"></i>Import Translation File
                    </button>
                    <button class="btn btn-outline-warning" disabled>
                        <i class="fas fa-language me-2"></i>Auto-translate Missing
                    </button>
                </div>
            </div>
        </div>
        
        <!-- Common Languages -->
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Quick Add Languages</h5>
            </div>
            <div class="card-body">
                <div class="row g-2">
                    <?php foreach ($common_languages as $name => $lang): ?>
                    <?php 
                    $is_installed = false;
                    foreach ($languages as $installed_lang) {
                        if ($installed_lang['code'] == $lang['code']) {
                            $is_installed = true;
                            break;
                        }
                    }
                    ?>
                    <div class="col-6">
                        <form method="POST" action="?action=language" class="d-grid">
                            <input type="hidden" name="action" value="add_language">
                            <input type="hidden" name="name" value="<?php echo $name; ?>">
                            <input type="hidden" name="code" value="<?php echo $lang['code']; ?>">
                            <input type="hidden" name="locale" value="<?php echo $lang['locale']; ?>">
                            <input type="hidden" name="status" value="active">
                            
                            <button type="submit" class="btn btn-outline-secondary btn-sm" 
                                    <?php echo $is_installed ? 'disabled' : ''; ?>>
                                <?php echo $name; ?>
                                <?php if ($is_installed): ?>
                                <span class="badge bg-success ms-1">✓</span>
                                <?php endif; ?>
                            </button>
                        </form>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add/Edit Language Modal -->
<div class="modal fade" id="languageModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><?php echo $edit_language ? 'Edit Language' : 'Add New Language'; ?></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" action="?action=language">
                <div class="modal-body">
                    <input type="hidden" name="action" value="<?php echo $edit_language ? 'update_language' : 'add_language'; ?>">
                    <?php if ($edit_language): ?>
                    <input type="hidden" name="id" value="<?php echo htmlspecialchars($edit_language['id']); ?>">
                    <?php endif; ?>
                    
                    <div class="mb-3">
                        <label for="name" class="form-label">Language Name *</label>
                        <input type="text" class="form-control" id="name" name="name" 
                               value="<?php echo htmlspecialchars($edit_language['name'] ?? ''); ?>" required>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="code" class="form-label">Language Code *</label>
                                <input type="text" class="form-control" id="code" name="code" 
                                       value="<?php echo htmlspecialchars($edit_language['code'] ?? ''); ?>" 
                                       maxlength="5" required>
                                <div class="form-text">e.g., en, es, fr</div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="locale" class="form-label">Locale</label>
                                <input type="text" class="form-control" id="locale" name="locale" 
                                       value="<?php echo htmlspecialchars($edit_language['locale'] ?? ''); ?>"
                                       placeholder="e.g., en_US, es_ES">
                                <div class="form-text">Language and country code</div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="status" class="form-label">Status *</label>
                        <select class="form-select" id="status" name="status" required>
                            <option value="active" <?php echo ($edit_language['status'] ?? '') == 'active' ? 'selected' : ''; ?>>Active</option>
                            <option value="inactive" <?php echo ($edit_language['status'] ?? '') == 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary"><?php echo $edit_language ? 'Update Language' : 'Add Language'; ?></button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php if ($_GET['edit'] ?? ''): ?>
<script>
document.addEventListener('DOMContentLoaded', function() {
    var languageModal = new bootstrap.Modal(document.getElementById('languageModal'));
    languageModal.show();
});
</script>
<?php endif; ?>