class PlacementTest {
    constructor() {
        this.questions = [];
        this.currentQuestionIndex = 0;
        this.userAnswers = [];
        this.totalQuestions = 20;
        
        this.init();
    }
    
    async init() {
        await this.loadQuestions();
        this.displayQuestion();
        this.setupEventListeners();
    }
    
    async loadQuestions() {
        // In a real implementation, this would fetch from your API
        // For demo purposes, we'll use sample questions
        this.questions = [
            {
                id: 1,
                text: "Choose the correct option: I ___ to the store yesterday.",
                options: {
                    a: "go",
                    b: "went",
                    c: "gone",
                    d: "going"
                },
                correct: "b",
                level: "A1"
            },
            // Add more questions...
        ];
    }
    
    displayQuestion() {
        if (this.currentQuestionIndex >= this.questions.length) {
            this.showResults();
            return;
        }
        
        const question = this.questions[this.currentQuestionIndex];
        document.getElementById('question-text').textContent = question.text;
        
        const optionsContainer = document.getElementById('options-container');
        optionsContainer.innerHTML = '';
        
        Object.entries(question.options).forEach(([key, value]) => {
            const optionDiv = document.createElement('div');
            optionDiv.className = 'form-check mb-2';
            optionDiv.innerHTML = `
                <input class="form-check-input" type="radio" name="answer" id="option-${key}" value="${key}">
                <label class="form-check-label" for="option-${key}">
                    ${value}
                </label>
            `;
            optionsContainer.appendChild(optionDiv);
        });
        
        // Update progress
        const progress = ((this.currentQuestionIndex + 1) / this.totalQuestions) * 100;
        document.querySelector('.progress-bar').style.width = `${progress}%`;
        document.getElementById('current-question').textContent = this.currentQuestionIndex + 1;
    }
    
    setupEventListeners() {
        document.getElementById('next-btn').addEventListener('click', () => {
            this.nextQuestion();
        });
        
        document.addEventListener('change', (e) => {
            if (e.target.name === 'answer') {
                document.getElementById('next-btn').disabled = false;
            }
        });
    }
    
    nextQuestion() {
        const selectedOption = document.querySelector('input[name="answer"]:checked');
        if (selectedOption) {
            this.userAnswers.push({
                questionId: this.questions[this.currentQuestionIndex].id,
                answer: selectedOption.value
            });
            
            this.currentQuestionIndex++;
            document.getElementById('next-btn').disabled = true;
            this.displayQuestion();
        }
    }
    
    showResults() {
        document.getElementById('question-container').classList.add('d-none');
        document.getElementById('results-container').classList.remove('d-none');
        
        // Calculate level based on answers
        const level = this.calculateLevel();
        document.getElementById('level-result').textContent = level;
        
        const descriptions = {
            'A1': 'Beginner level. You can understand and use familiar everyday expressions.',
            'A2': 'Elementary level. You can communicate in simple and routine tasks.',
            'B1': 'Intermediate level. You can deal with most situations while traveling.',
            'B2': 'Upper Intermediate level. You can interact with fluency and spontaneity.',
            'C1': 'Advanced level. You can use language flexibly and effectively.',
            'C2': 'Proficient level. You can express yourself spontaneously and precisely.'
        };
        
        document.getElementById('level-description').textContent = descriptions[level];
        
        // Save results to server (would be implemented in backend)
        this.saveResults(level);
    }
    
    calculateLevel() {
        // Simplified level calculation
        // In practice, this would be more sophisticated
        const scores = ['A1', 'A2', 'B1', 'B2', 'C1', 'C2'];
        const correctAnswers = this.userAnswers.filter(answer => {
            const question = this.questions.find(q => q.id === answer.questionId);
            return question && question.correct === answer.answer;
        }).length;
        
        const percentage = (correctAnswers / this.questions.length) * 100;
        
        if (percentage < 20) return 'A1';
        if (percentage < 40) return 'A2';
        if (percentage < 60) return 'B1';
        if (percentage < 80) return 'B2';
        if (percentage < 95) return 'C1';
        return 'C2';
    }
    
    async saveResults(level) {
        // Implementation for saving results to backend
        try {
            const response = await fetch('../app/api/save-test-results.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    level: level,
                    answers: this.userAnswers
                })
            });
            
            if (response.ok) {
                console.log('Results saved successfully');
            }
        } catch (error) {
            console.error('Error saving results:', error);
        }
    }
}

// Initialize the test when page loads
document.addEventListener('DOMContentLoaded', () => {
    new PlacementTest();
});