<?php
/**
 * Simturax English Learning Platform - Diagnostic Script
 * This script will help identify the cause of the 500 Internal Server Error
 * DELETE THIS FILE AFTER USE FOR SECURITY
 */

// Prevent caching
header("Cache-Control: no-cache, must-revalidate");
header("Expires: Sat, 26 Jul 1997 05:00:00 GMT");

// Error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Site Diagnostic - Simturax</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .status-ok { background-color: #d4edda; color: #155724; }
        .status-warning { background-color: #fff3cd; color: #856404; }
        .status-error { background-color: #f8d7da; color: #721c24; }
        .card { margin-bottom: 20px; }
        .test-result { padding: 10px; border-radius: 5px; margin: 5px 0; }
    </style>
</head>
<body>
    <div class="container mt-5">
        <div class="row">
            <div class="col-12">
                <h1 class="text-center mb-4">Simturax Site Diagnostic</h1>
                
                <?php
                // Test results array
                $testResults = [];

                // Function to add test result
                function addTestResult(&$results, $test, $status, $message, $details = '') {
                    $results[] = [
                        'test' => $test,
                        'status' => $status,
                        'message' => $message,
                        'details' => $details
                    ];
                }

                // 1. PHP Version Check
                $phpVersion = phpversion();
                $minPhpVersion = '7.4';
                if (version_compare($phpVersion, $minPhpVersion, '>=')) {
                    addTestResult($testResults, 'PHP Version', 'ok', "PHP $phpVersion - OK", "Meets minimum requirement: $minPhpVersion");
                } else {
                    addTestResult($testResults, 'PHP Version', 'error', "PHP $phpVersion - TOO OLD", "Minimum required: $minPhpVersion");
                }

                // 2. PHP Extensions Check
                $requiredExtensions = ['pdo', 'pdo_mysql', 'session', 'json', 'mbstring', 'openssl'];
                $missingExtensions = [];
                foreach ($requiredExtensions as $ext) {
                    if (!extension_loaded($ext)) {
                        $missingExtensions[] = $ext;
                    }
                }
                if (empty($missingExtensions)) {
                    addTestResult($testResults, 'PHP Extensions', 'ok', 'All required extensions loaded', implode(', ', $requiredExtensions));
                } else {
                    addTestResult($testResults, 'PHP Extensions', 'error', 'Missing extensions: ' . implode(', ', $missingExtensions), 'Contact hosting provider to enable these');
                }

                // 3. Directory Permissions Check
                $directoriesToCheck = [
                    '.' => 'Root Directory',
                    'admin' => 'Admin Directory', 
                    'app' => 'App Directory',
                    'assets' => 'Assets Directory',
                    'assets/css' => 'CSS Directory',
                    'assets/js' => 'JS Directory'
                ];
                
                $permissionIssues = [];
                foreach ($directoriesToCheck as $dir => $desc) {
                    if (!is_dir($dir) && $dir !== '.') {
                        $permissionIssues[] = "$desc: Directory doesn't exist";
                    } elseif (!is_readable($dir)) {
                        $permissionIssues[] = "$desc: Not readable";
                    } elseif (!is_writable($dir) && in_array($dir, ['.', 'assets', 'app'])) {
                        $permissionIssues[] = "$desc: Not writable";
                    }
                }
                
                if (empty($permissionIssues)) {
                    addTestResult($testResults, 'Directory Permissions', 'ok', 'All directories accessible', 'Required permissions: 755 for directories, 644 for files');
                } else {
                    addTestResult($testResults, 'Directory Permissions', 'error', 'Permission issues found', implode('<br>', $permissionIssues));
                }

                // 4. .htaccess File Check
                $htaccessFiles = [
                    '.htaccess' => 'Root',
                    'admin/.htaccess' => 'Admin',
                    'app/.htaccess' => 'App',
                    'assets/.htaccess' => 'Assets'
                ];
                
                $htaccessStatus = [];
                foreach ($htaccessFiles as $file => $location) {
                    if (file_exists($file)) {
                        $content = file_get_contents($file);
                        if (strpos($content, 'RewriteEngine') !== false) {
                            $htaccessStatus[] = "$location: OK";
                        } else {
                            $htaccessStatus[] = "$location: Missing RewriteEngine";
                        }
                    } else {
                        $htaccessStatus[] = "$location: File missing";
                    }
                }
                
                addTestResult($testResults, '.htaccess Files', 'warning', 'Mod_rewrite configuration', implode('<br>', $htaccessStatus));

                // 5. Database Connection Test
                try {
                    $db_host = "localhost";
                    $db_name = "techoisi_simturax";
                    $db_user = "techoisi_simturax";
                    $db_pass = "0670445472@z";
                    
                    $pdo = new PDO("mysql:host=$db_host;dbname=$db_name", $db_user, $db_pass);
                    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
                    
                    // Test query
                    $stmt = $pdo->query("SELECT 1");
                    if ($stmt) {
                        addTestResult($testResults, 'Database Connection', 'ok', 'Database connection successful', "Connected to: $db_name");
                    }
                } catch (PDOException $e) {
                    addTestResult($testResults, 'Database Connection', 'error', 'Database connection failed', $e->getMessage());
                }

                // 6. Required Files Check
                $requiredFiles = [
                    'index.php',
                    'admin/index.php', 
                    'app/config/database.php',
                    'assets/css/style.css'
                ];
                
                $missingFiles = [];
                foreach ($requiredFiles as $file) {
                    if (!file_exists($file)) {
                        $missingFiles[] = $file;
                    }
                }
                
                if (empty($missingFiles)) {
                    addTestResult($testResults, 'Required Files', 'ok', 'All core files present', count($requiredFiles) . ' files checked');
                } else {
                    addTestResult($testResults, 'Required Files', 'error', 'Missing files detected', implode('<br>', $missingFiles));
                }

                // 7. PHP Configuration Check
                $phpConfig = [
                    'memory_limit' => '128M',
                    'max_execution_time' => '30',
                    'upload_max_filesize' => '10M',
                    'post_max_size' => '10M'
                ];
                
                $configIssues = [];
                foreach ($phpConfig as $setting => $recommended) {
                    $current = ini_get($setting);
                    if ($current < $recommended && is_numeric(str_replace('M', '', $current))) {
                        $configIssues[] = "$setting: $current (Recommended: $recommended)";
                    }
                }
                
                if (empty($configIssues)) {
                    addTestResult($testResults, 'PHP Configuration', 'ok', 'PHP settings adequate', 'Current settings meet recommendations');
                } else {
                    addTestResult($testResults, 'PHP Configuration', 'warning', 'PHP settings may need adjustment', implode('<br>', $configIssues));
                }

                // 8. mod_rewrite Test
                $modRewriteEnabled = false;
                if (function_exists('apache_get_modules')) {
                    $modRewriteEnabled = in_array('mod_rewrite', apache_get_modules());
                } else {
                    // Alternative check for non-Apache servers
                    $modRewriteEnabled = (strpos($_SERVER['SERVER_SOFTWARE'], 'Apache') !== false) ? 'Unknown' : 'N/A (Not Apache)';
                }
                
                if ($modRewriteEnabled === true) {
                    addTestResult($testResults, 'mod_rewrite', 'ok', 'mod_rewrite is enabled', 'URL rewriting should work');
                } elseif ($modRewriteEnabled === 'N/A (Not Apache)') {
                    addTestResult($testResults, 'mod_rewrite', 'warning', 'Not running Apache', 'URL rewriting depends on server type');
                } else {
                    addTestResult($testResults, 'mod_rewrite', 'error', 'mod_rewrite not enabled', 'Contact hosting provider to enable mod_rewrite');
                }

                // 9. Error Log Check
                $errorLog = ini_get('error_log');
                $displayErrors = ini_get('display_errors');
                
                $errorLogInfo = "Error Log: " . ($errorLog ? $errorLog : 'Default') . "<br>";
                $errorLogInfo .= "Display Errors: " . ($displayErrors ? 'On' : 'Off') . "<br>";
                $errorLogInfo .= "Error Reporting: " . error_reporting();
                
                addTestResult($testResults, 'Error Reporting', 'warning', 'Error configuration', $errorLogInfo);

                // Display Results
                foreach ($testResults as $result) {
                    $statusClass = 'status-' . $result['status'];
                    echo "
                    <div class='card'>
                        <div class='card-header d-flex justify-content-between align-items-center'>
                            <h5 class='mb-0'>{$result['test']}</h5>
                            <span class='badge " . ($result['status'] == 'ok' ? 'bg-success' : ($result['status'] == 'warning' ? 'bg-warning' : 'bg-danger')) . "'>
                                " . strtoupper($result['status']) . "
                            </span>
                        </div>
                        <div class='card-body'>
                            <p class='card-text'><strong>{$result['message']}</strong></p>
                            <div class='test-result $statusClass'>
                                <small>{$result['details']}</small>
                            </div>
                        </div>
                    </div>";
                }

                // Additional Server Information
                ?>
                
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Server Information</h5>
                    </div>
                    <div class="card-body">
                        <table class="table table-striped">
                            <tr>
                                <td><strong>Server Software</strong></td>
                                <td><?php echo $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown'; ?></td>
                            </tr>
                            <tr>
                                <td><strong>PHP Version</strong></td>
                                <td><?php echo phpversion(); ?></td>
                            </tr>
                            <tr>
                                <td><strong>Server Name</strong></td>
                                <td><?php echo $_SERVER['SERVER_NAME'] ?? 'Unknown'; ?></td>
                            </tr>
                            <tr>
                                <td><strong>Document Root</strong></td>
                                <td><?php echo $_SERVER['DOCUMENT_ROOT'] ?? 'Unknown'; ?></td>
                            </tr>
                            <tr>
                                <td><strong>Client IP</strong></td>
                                <td><?php echo $_SERVER['REMOTE_ADDR'] ?? 'Unknown'; ?></td>
                            </tr>
                            <tr>
                                <td><strong>Request Method</strong></td>
                                <td><?php echo $_SERVER['REQUEST_METHOD'] ?? 'Unknown'; ?></td>
                            </tr>
                        </table>
                    </div>
                </div>

                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Quick Fixes</h5>
                    </div>
                    <div class="card-body">
                        <h6>If you're getting 500 errors:</h6>
                        <ol>
                            <li>Check the error logs in cPanel</li>
                            <li>Ensure all file permissions are set to 644 (files) and 755 (directories)</li>
                            <li>Verify database credentials in app/config/database.php</li>
                            <li>Temporarily rename .htaccess to test if it's the cause</li>
                            <li>Contact hosting provider if mod_rewrite is not enabled</li>
                        </ol>
                        
                        <div class="alert alert-warning">
                            <strong>Security Notice:</strong> Delete this diagnostic.php file after use!
                        </div>
                        
                        <button class="btn btn-danger" onclick="if(confirm('Are you sure you want to delete this diagnostic file?')) { window.location.href = '?delete=1'; }">
                            Delete This Diagnostic File
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

<?php
// Handle file deletion
if (isset($_GET['delete']) && $_GET['delete'] == 1) {
    if (unlink(__FILE__)) {
        echo "<script>alert('Diagnostic file deleted successfully!'); window.close();</script>";
    } else {
        echo "<script>alert('Error deleting file. Please delete manually.');</script>";
    }
    exit;
}
?>