<?php
session_start();

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: /');
    exit;
}

// Include database
require_once '../../app/config/database.php';
$database = new Database();
$db = $database->getConnection();

// Get user data
$user_id = $_SESSION['user_id'];
$query = "SELECT * FROM users WHERE id = ?";
$stmt = $db->prepare($query);
$stmt->execute([$user_id]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($_POST['action'] === 'update_profile') {
        $first_name = $_POST['first_name'] ?? '';
        $last_name = $_POST['last_name'] ?? '';
        $email = $_POST['email'] ?? '';
        $username = $_POST['username'] ?? '';
        
        // Handle profile image upload
        $profile_image = $user['profile_image'];
        if (isset($_FILES['profile_image']) && $_FILES['profile_image']['error'] === 0) {
            $uploadDir = '../../assets/uploads/profiles/';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }
            
            $fileExt = strtolower(pathinfo($_FILES['profile_image']['name'], PATHINFO_EXTENSION));
            $fileName = 'profile_' . $user_id . '_' . time() . '.' . $fileExt;
            $filePath = $uploadDir . $fileName;
            
            if (move_uploaded_file($_FILES['profile_image']['tmp_name'], $filePath)) {
                $profile_image = '/assets/uploads/profiles/' . $fileName;
            }
        }
        
        $query = "UPDATE users SET first_name = ?, last_name = ?, email = ?, username = ?, profile_image = ?, updated_at = NOW() WHERE id = ?";
        $stmt = $db->prepare($query);
        if ($stmt->execute([$first_name, $last_name, $email, $username, $profile_image, $user_id])) {
            $success = "Profile updated successfully!";
            // Refresh user data
            $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
            $stmt->execute([$user_id]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
        } else {
            $error = "Error updating profile. Please try again.";
        }
    }
    
    if ($_POST['action'] === 'change_password') {
        $current_password = $_POST['current_password'] ?? '';
        $new_password = $_POST['new_password'] ?? '';
        $confirm_password = $_POST['confirm_password'] ?? '';
        
        // Verify current password
        if (password_verify($current_password, $user['password'])) {
            if ($new_password === $confirm_password) {
                if (strlen($new_password) >= 6) {
                    $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                    $query = "UPDATE users SET password = ?, updated_at = NOW() WHERE id = ?";
                    $stmt = $db->prepare($query);
                    if ($stmt->execute([$hashed_password, $user_id])) {
                        $success_password = "Password changed successfully!";
                    } else {
                        $error_password = "Error changing password. Please try again.";
                    }
                } else {
                    $error_password = "New password must be at least 6 characters long.";
                }
            } else {
                $error_password = "New passwords do not match.";
            }
        } else {
            $error_password = "Current password is incorrect.";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Profile Settings - Simturax</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .profile-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 40px 0;
            margin-bottom: 30px;
        }
        
        .profile-card {
            border: none;
            border-radius: 15px;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
            margin-bottom: 30px;
        }
        
        .profile-image {
            width: 150px;
            height: 150px;
            border-radius: 50%;
            object-fit: cover;
            border: 5px solid white;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.2);
        }
        
        .nav-pills .nav-link {
            border-radius: 10px;
            margin: 5px 0;
            padding: 12px 20px;
        }
        
        .nav-pills .nav-link.active {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
    </style>
</head>
<body>
    <!-- Navigation (same as dashboard) -->
    <nav class="navbar navbar-expand-lg navbar-dark" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);">
        <div class="container">
            <a class="navbar-brand fw-bold" href="/">
                <i class="fas fa-graduation-cap me-2"></i>Simturax
            </a>
            <div class="d-flex align-items-center">
                <a href="/dashboard" class="btn btn-outline-light btn-sm me-2">
                    <i class="fas fa-arrow-left me-1"></i>Back to Dashboard
                </a>
                <div class="dropdown">
                    <a href="#" class="d-flex align-items-center text-white text-decoration-none dropdown-toggle" data-bs-toggle="dropdown">
                        <img src="<?php echo $user['profile_image'] ?? '/assets/images/default-avatar.png'; ?>" 
                             class="rounded-circle me-2" width="32" height="32" style="object-fit: cover;">
                        <span><?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?></span>
                    </a>
                    <div class="dropdown-menu dropdown-menu-end">
                        <a class="dropdown-item" href="/dashboard">
                            <i class="fas fa-tachometer-alt me-2"></i>Dashboard
                        </a>
                        <a class="dropdown-item" href="/logout.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Logout
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </nav>

    <div class="profile-header">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <h1 class="display-5 fw-bold">Profile Settings</h1>
                    <p class="lead mb-0">Manage your account information and preferences</p>
                </div>
                <div class="col-md-4 text-center">
                    <img src="<?php echo $user['profile_image'] ?? '/assets/images/default-avatar.png'; ?>" 
                         class="profile-image" alt="Profile">
                </div>
            </div>
        </div>
    </div>

    <div class="container">
        <div class="row">
            <!-- Sidebar Navigation -->
            <div class="col-lg-3 mb-4">
                <div class="profile-card p-4">
                    <ul class="nav nav-pills flex-column">
                        <li class="nav-item">
                            <a class="nav-link active" href="#profile" data-bs-toggle="tab">
                                <i class="fas fa-user me-2"></i>Profile Information
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="#password" data-bs-toggle="tab">
                                <i class="fas fa-lock me-2"></i>Password & Security
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="#preferences" data-bs-toggle="tab">
                                <i class="fas fa-cog me-2"></i>Preferences
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="#notifications" data-bs-toggle="tab">
                                <i class="fas fa-bell me-2"></i>Notifications
                            </a>
                        </li>
                    </ul>
                </div>
            </div>

            <!-- Main Content -->
            <div class="col-lg-9">
                <div class="tab-content">
                    <!-- Profile Information Tab -->
                    <div class="tab-pane fade show active" id="profile">
                        <div class="profile-card p-4">
                            <h4 class="mb-4">Profile Information</h4>
                            
                            <?php if (isset($success)): ?>
                            <div class="alert alert-success alert-dismissible fade show" role="alert">
                                <?php echo $success; ?>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                            <?php endif; ?>
                            
                            <?php if (isset($error)): ?>
                            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                                <?php echo $error; ?>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                            <?php endif; ?>
                            
                            <form method="POST" enctype="multipart/form-data">
                                <input type="hidden" name="action" value="update_profile">
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="first_name" class="form-label">First Name</label>
                                        <input type="text" class="form-control" id="first_name" name="first_name" 
                                               value="<?php echo htmlspecialchars($user['first_name'] ?? ''); ?>" required>
                                    </div>
                                    <div class="col-md-6 mb-3">
                                        <label for="last_name" class="form-label">Last Name</label>
                                        <input type="text" class="form-control" id="last_name" name="last_name" 
                                               value="<?php echo htmlspecialchars($user['last_name'] ?? ''); ?>" required>
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="email" class="form-label">Email Address</label>
                                        <input type="email" class="form-control" id="email" name="email" 
                                               value="<?php echo htmlspecialchars($user['email'] ?? ''); ?>" required>
                                    </div>
                                    <div class="col-md-6 mb-3">
                                        <label for="username" class="form-label">Username</label>
                                        <input type="text" class="form-control" id="username" name="username" 
                                               value="<?php echo htmlspecialchars($user['username'] ?? ''); ?>">
                                    </div>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="profile_image" class="form-label">Profile Image</label>
                                    <input type="file" class="form-control" id="profile_image" name="profile_image" 
                                           accept="image/*">
                                    <div class="form-text">Recommended size: 150x150 pixels. Max file size: 2MB.</div>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="bio" class="form-label">Bio</label>
                                    <textarea class="form-control" id="bio" name="bio" rows="4" 
                                              placeholder="Tell us about yourself..."><?php echo htmlspecialchars($user['bio'] ?? ''); ?></textarea>
                                </div>
                                
                                <button type="submit" class="btn btn-primary">Update Profile</button>
                            </form>
                        </div>
                    </div>

                    <!-- Password & Security Tab -->
                    <div class="tab-pane fade" id="password">
                        <div class="profile-card p-4">
                            <h4 class="mb-4">Password & Security</h4>
                            
                            <?php if (isset($success_password)): ?>
                            <div class="alert alert-success alert-dismissible fade show" role="alert">
                                <?php echo $success_password; ?>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                            <?php endif; ?>
                            
                            <?php if (isset($error_password)): ?>
                            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                                <?php echo $error_password; ?>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                            <?php endif; ?>
                            
                            <form method="POST">
                                <input type="hidden" name="action" value="change_password">
                                
                                <div class="mb-3">
                                    <label for="current_password" class="form-label">Current Password</label>
                                    <input type="password" class="form-control" id="current_password" name="current_password" required>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="new_password" class="form-label">New Password</label>
                                    <input type="password" class="form-control" id="new_password" name="new_password" required>
                                    <div class="form-text">Password must be at least 6 characters long.</div>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="confirm_password" class="form-label">Confirm New Password</label>
                                    <input type="password" class="form-control" id="confirm_password" name="confirm_password" required>
                                </div>
                                
                                <button type="submit" class="btn btn-primary">Change Password</button>
                            </form>
                            
                            <hr class="my-4">
                            
                            <h6 class="mb-3">Security Settings</h6>
                            <div class="form-check form-switch mb-2">
                                <input class="form-check-input" type="checkbox" id="twoFactor" checked>
                                <label class="form-check-label" for="twoFactor">Two-factor authentication</label>
                            </div>
                            <div class="form-check form-switch mb-2">
                                <input class="form-check-input" type="checkbox" id="loginAlerts" checked>
                                <label class="form-check-label" for="loginAlerts">Login alerts</label>
                            </div>
                        </div>
                    </div>

                    <!-- Preferences Tab -->
                    <div class="tab-pane fade" id="preferences">
                        <div class="profile-card p-4">
                            <h4 class="mb-4">Learning Preferences</h4>
                            
                            <form>
                                <div class="mb-3">
                                    <label class="form-label">Daily Learning Goal</label>
                                    <select class="form-select" id="daily_goal">
                                        <option value="15">15 minutes</option>
                                        <option value="30" selected>30 minutes</option>
                                        <option value="45">45 minutes</option>
                                        <option value="60">60 minutes</option>
                                    </select>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label">Preferred Learning Style</label>
                                    <select class="form-select" id="learning_style">
                                        <option value="visual">Visual</option>
                                        <option value="auditory">Auditory</option>
                                        <option value="kinesthetic" selected>Kinesthetic</option>
                                        <option value="mixed">Mixed</option>
                                    </select>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label">Difficulty Level</label>
                                    <select class="form-select" id="difficulty">
                                        <option value="easy">Easy</option>
                                        <option value="medium" selected>Medium</option>
                                        <option value="hard">Hard</option>
                                    </select>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label">Study Reminders</label>
                                    <div class="form-check mb-2">
                                        <input class="form-check-input" type="checkbox" id="reminder_daily" checked>
                                        <label class="form-check-label" for="reminder_daily">Daily study reminders</label>
                                    </div>
                                    <div class="form-check mb-2">
                                        <input class="form-check-input" type="checkbox" id="reminder_weekly" checked>
                                        <label class="form-check-label" for="reminder_weekly">Weekly progress reports</label>
                                    </div>
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="reminder_achievements">
                                        <label class="form-check-label" for="reminder_achievements">Achievement notifications</label>
                                    </div>
                                </div>
                                
                                <button type="button" class="btn btn-primary">Save Preferences</button>
                            </form>
                        </div>
                    </div>

                    <!-- Notifications Tab -->
                    <div class="tab-pane fade" id="notifications">
                        <div class="profile-card p-4">
                            <h4 class="mb-4">Notification Settings</h4>
                            
                            <form>
                                <h6 class="mb-3">Email Notifications</h6>
                                <div class="form-check mb-2">
                                    <input class="form-check-input" type="checkbox" id="email_lessons" checked>
                                    <label class="form-check-label" for="email_lessons">New lessons and courses</label>
                                </div>
                                <div class="form-check mb-2">
                                    <input class="form-check-input" type="checkbox" id="email_progress" checked>
                                    <label class="form-check-label" for="email_progress">Progress updates</label>
                                </div>
                                <div class="form-check mb-2">
                                    <input class="form-check-input" type="checkbox" id="email_announcements">
                                    <label class="form-check-label" for="email_announcements">System announcements</label>
                                </div>
                                
                                <hr class="my-4">
                                
                                <h6 class="mb-3">Push Notifications</h6>
                                <div class="form-check mb-2">
                                    <input class="form-check-input" type="checkbox" id="push_reminders" checked>
                                    <label class="form-check-label" for="push_reminders">Study reminders</label>
                                </div>
                                <div class="form-check mb-2">
                                    <input class="form-check-input" type="checkbox" id="push_achievements" checked>
                                    <label class="form-check-label" for="push_achievements">Achievement unlocked</label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="push_messages">
                                    <label class="form-check-label" for="push_messages">Instructor messages</label>
                                </div>
                                
                                <hr class="my-4">
                                
                                <h6 class="mb-3">SMS Notifications</h6>
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="sms_important">
                                    <label class="form-check-label" for="sms_important">Important updates only</label>
                                </div>
                                
                                <button type="button" class="btn btn-primary mt-3">Save Notification Settings</button>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Tab functionality
        document.addEventListener('DOMContentLoaded', function() {
            // Show first tab by default
            const triggerEl = document.querySelector('#profile-tab');
            const tab = new bootstrap.Tab(triggerEl);
            
            // Handle tab clicks
            const tabLinks = document.querySelectorAll('a[data-bs-toggle="tab"]');
            tabLinks.forEach(link => {
                link.addEventListener('click', function(e) {
                    e.preventDefault();
                    const tab = new bootstrap.Tab(this);
                    tab.show();
                });
            });
        });
    </script>
</body>
</html>